#!/bin/bash

# Cattr cPanel Deployment Script
# Run this script after uploading files to your cPanel hosting
# Usage: bash cpanelDeploy.sh

set -e

echo "========================================="
echo "  Cattr cPanel Deployment Script"
echo "========================================="
echo ""

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# Check if we're in the right directory
if [ ! -f "artisan" ]; then
    echo -e "${RED}Error: artisan file not found. Please run this script from the Laravel root directory.${NC}"
    exit 1
fi

echo -e "${YELLOW}Step 1: Checking PHP version...${NC}"
php -v
echo ""

echo -e "${YELLOW}Step 2: Checking required PHP extensions...${NC}"
php -m | grep -E "gd|zip|mbstring|pdo_mysql|openssl" || echo -e "${RED}Some extensions may be missing. Check cPanel PHP Extensions.${NC}"
echo ""

echo -e "${YELLOW}Step 3: Setting up environment file...${NC}"
if [ ! -f ".env" ]; then
    if [ -f ".env.example" ]; then
        cp .env.example .env
        echo -e "${GREEN}✓ Created .env from .env.example${NC}"
    else
        echo -e "${RED}✗ .env.example not found!${NC}"
        exit 1
    fi
else
    echo -e "${GREEN}✓ .env file already exists${NC}"
fi
echo ""

echo -e "${YELLOW}Step 4: Installing/Updating Composer dependencies...${NC}"
if command -v composer &> /dev/null; then
    composer install --optimize-autoloader --no-dev
    echo -e "${GREEN}✓ Composer dependencies installed${NC}"
else
    echo -e "${RED}✗ Composer not found. Please install dependencies manually.${NC}"
fi
echo ""

echo -e "${YELLOW}Step 5: Generating application key...${NC}"
php artisan key:generate --force
echo -e "${GREEN}✓ Application key generated${NC}"
echo ""

echo -e "${YELLOW}Step 6: Creating storage directories...${NC}"
mkdir -p storage/app/screenshots
mkdir -p storage/app/screenshots/thumbs
mkdir -p storage/framework/cache
mkdir -p storage/framework/sessions
mkdir -p storage/framework/views
mkdir -p storage/logs
mkdir -p bootstrap/cache
echo -e "${GREEN}✓ Storage directories created${NC}"
echo ""

echo -e "${YELLOW}Step 7: Setting directory permissions...${NC}"
chmod -R 775 storage
chmod -R 775 bootstrap/cache
echo -e "${GREEN}✓ Permissions set${NC}"
echo ""

echo -e "${YELLOW}Step 8: Creating database tables...${NC}"
read -p "Do you want to run migrations? (y/n): " -n 1 -r
echo ""
if [[ $REPLY =~ ^[Yy]$ ]]; then
    php artisan migrate --force
    echo -e "${GREEN}✓ Database migrated${NC}"
else
    echo -e "${YELLOW}⊘ Skipped database migration${NC}"
fi
echo ""

echo -e "${YELLOW}Step 9: Running initial seeders...${NC}"
read -p "Do you want to seed the database with initial data? (y/n): " -n 1 -r
echo ""
if [[ $REPLY =~ ^[Yy]$ ]]; then
    php artisan db:seed --class=InitialSeeder --force
    echo -e "${GREEN}✓ Initial data seeded${NC}"
else
    echo -e "${YELLOW}⊘ Skipped seeding${NC}"
fi
echo ""

echo -e "${YELLOW}Step 10: Creating admin user...${NC}"
read -p "Do you want to create an admin user? (y/n): " -n 1 -r
echo ""
if [[ $REPLY =~ ^[Yy]$ ]]; then
    php artisan cattr:make:admin
    echo -e "${GREEN}✓ Admin user creation initiated${NC}"
else
    echo -e "${YELLOW}⊘ Skipped admin creation${NC}"
fi
echo ""

echo -e "${YELLOW}Step 11: Optimizing application...${NC}"
php artisan config:cache
php artisan route:cache
php artisan view:cache
echo -e "${GREEN}✓ Application optimized${NC}"
echo ""

echo -e "${YELLOW}Step 12: Creating queue jobs table...${NC}"
read -p "Do you want to create the queue jobs table? (y/n): " -n 1 -r
echo ""
if [[ $REPLY =~ ^[Yy]$ ]]; then
    php artisan queue:table
    php artisan migrate --force
    echo -e "${GREEN}✓ Queue table created${NC}"
else
    echo -e "${YELLOW}⊘ Skipped queue table creation${NC}"
fi
echo ""

echo -e "${YELLOW}Step 13: Building frontend assets...${NC}"
if command -v npm &> /dev/null; then
    read -p "Do you want to build frontend assets? (y/n): " -n 1 -r
    echo ""
    if [[ $REPLY =~ ^[Yy]$ ]]; then
        npm install
        npm run prod
        echo -e "${GREEN}✓ Frontend assets built${NC}"
    else
        echo -e "${YELLOW}⊘ Skipped frontend build${NC}"
    fi
else
    echo -e "${YELLOW}⊘ npm not found. Please build assets locally and upload.${NC}"
fi
echo ""

echo "========================================="
echo -e "${GREEN}✓ Deployment Complete!${NC}"
echo "========================================="
echo ""
echo -e "${YELLOW}IMPORTANT NEXT STEPS:${NC}"
echo ""
echo "1. Update your .env file with production settings:"
echo "   - Set APP_ENV=production"
echo "   - Set APP_DEBUG=false"
echo "   - Configure database credentials"
echo "   - Set APP_URL to your domain"
echo "   - Set QUEUE_CONNECTION=database"
echo ""
echo "2. Set up a cron job in cPanel for queue processing:"
echo "   Command: * * * * * cd $(pwd) && /usr/bin/php artisan queue:work --stop-when-empty >> /dev/null 2>&1"
echo ""
echo "3. Configure your web server document root to point to:"
echo "   $(pwd)/public"
echo ""
echo "4. If using subdirectory, create .htaccess in public_html:"
echo "   <IfModule mod_rewrite.c>"
echo "       RewriteEngine On"
echo "       RewriteRule ^(.*)$ /path-to-app/public/\$1 [L]"
echo "   </IfModule>"
echo ""
echo "5. Clear caches after .env changes:"
echo "   php artisan config:cache"
echo ""
echo "6. Test the application at your domain"
echo ""
echo "========================================="
echo -e "${GREEN}Need help? Visit: https://docs.cattr.app${NC}"
echo "========================================="
